/* --------------------------------------------------------------
  DuplicateStyleModal.js 2019-06-07
  Gambio GmbH
  http://www.gambio.de
  Copyright (c) 2019 Gambio GmbH
  Released under the GNU General Public License (Version 2)
  [http://www.gnu.org/licenses/gpl-2.0.html]
  --------------------------------------------------------------*/


/* globals -Modal */

'use strict';

import Modal from '../libs/Modal'; 
import EventsEmitter from '../libs/EventsEmitter';
import StyleApi from '../api/StyleApi';

/**
 * Duplicate Style Modal Controller
 *
 * This controller handles the style duplication operation.
 */
export default class DuplicateStyleModal {
	/**
	 * Class Constructor
	 *
	 * @param {String} styleName Style name to be duplicated.
	 */
	constructor(styleName) {
		StyleEdit.Validator.isString(styleName);
		
		/**
		 * The modal selector.
		 *
		 * @type {jQuery}
		 */
		this.$modal = null;
		
		/**
		 * Style name to be duplicated.
		 *
		 * @type {String}
		 */
		this.styleName = styleName;
	}
	
	/**
	 * Initialize Controller
	 */
	initialize() {
		const data = {
			title_duplicate_style_modal: StyleEdit.Language.translate('title_duplicate_style_modal', 'style_edit'),
			placeholder_style_name: StyleEdit.Language.translate('placeholder_style_name', 'style_edit'),
			option_create: StyleEdit.Language.translate('option_create', 'style_edit'),
			option_cancel: StyleEdit.Language.translate('option_cancel', 'style_edit')
		};
		
		this.$modal = Modal.show($('#duplicate-style-modal-template'), data);
		
		// Focus style-name input field.
		this.$modal.find('input[name="style-name"]').focus();
		
		this._bindEventHandlers();
		
		$.material.init();
		
		EventsEmitter.triggerControllerInitialized(this.$modal, ['DuplicateStyleModal']);
	}
	
	/**
	 * Destroy Controller
	 */
	destroy() {
		Modal.hide(this.$modal);
		EventsEmitter.triggerControllerDestroyed(this.$modal, ['DuplicateStyleModal']);
	}
	
	/**
	 * Event: Duplicate Button Click
	 *
	 * @param {jQuery.Event} event
	 *
	 * @private
	 */
	_onClickCreate(event) {
		const $element = $(event.currentTarget);
		const $newStyleName = this.$modal.find('input[name="style-name"]');
		
		$newStyleName.parent().removeClass('has-error');
		
		if ($newStyleName.val() === '') {
			$newStyleName.parent().addClass('has-error');
			$newStyleName.focus();
			return;
		}
		
		StyleApi.duplicate(this.styleName, $newStyleName.val()).done(response => {
			EventsEmitter.triggerStyleCreated($element, [response]);
			this.destroy();
		});
	}
	
	/**
	 * Event: Cancel Button Click
	 *
	 * @private
	 */
	_onClickCancel() {
		this.destroy();
	}
	
	/**
	 * Bind modal event handlers.
	 *
	 * @private
	 */
	_bindEventHandlers() {
		this.$modal
			.on('click', '.btn.create', event => this._onClickCreate(event))
			.on('click', '.btn.cancel', () => this._onClickCancel());
	}
}
